/*
 * $Source: /usr/cvsroot/melati/melati/src/main/java/org/melati/test/CharData.java,v $
 * $Revision: 1.9 $
 *
 * Copyright (C) 2003 Jim Wright
 *
 * Part of Melati (http://melati.org), a framework for the rapid
 * development of clean, maintainable web applications.
 *
 * Melati is free software; Permission is granted to copy, distribute
 * and/or modify this software under the terms either:
 *
 * a) the GNU General Public License as published by the Free Software
 *    Foundation; either version 2 of the License, or (at your option)
 *    any later version,
 *
 *    or
 *
 * b) any version of the Melati Software License, as published
 *    at http://melati.org
 *
 * You should have received a copy of the GNU General Public License and
 * the Melati Software License along with this program;
 * if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA to obtain the
 * GNU General Public License and visit http://melati.org to obtain the
 * Melati Software License.
 *
 * Feel free to contact the Developers of Melati (http://melati.org),
 * if you would like to work out a different arrangement than the options
 * outlined here.  It is our intention to allow Melati to be used by as
 * wide an audience as possible.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Contact details for copyright holder:
 *
 *     Jim Wright <jimw At paneris.org>
 *     Bohemian Enterprise
 *     Predmerice nad Jizerou 77
 *     294 74
 *     Mlada Boleslav
 *     Czech Republic
 */

package org.melati.test;

import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CharsetEncoder;
import java.util.Iterator;

import org.melati.Melati;

/**
 * Provides character data for test servlets.
 */
public final class CharData {

  private CharData() {}

  /**
   * @return an Iterator of Items
   */
  public static Iterator<Item> getItems() {
    return new Iterator<Item>() {

        int i = 0;

        public boolean hasNext() {
          return i < charData.length;
        }

        public Item next() {
          return new Item(i++);
        }

        public void remove() throws UnsupportedOperationException {
          throw new UnsupportedOperationException();
        }
      };
  }

  /** A data item. **/
  public static class Item {

    private int i;

    /**
     * Constructor.
     * @param i
     */
    public Item(int i) {
      this.i = i;
    }

    /**
     * @return next character
     */
    public final String getChar() {
      return charData[i][0];
    }

    /**
     * @return description of the character
     */
    public final String getDescription() {
      return charData[i][1];
    }

    /**
     * @return the numeric entity for the character
     */
    public final String getReference() {
      return "&#x" + Integer.toHexString(getChar().charAt(0)) + ";";
    }

    /**
     * @param melati current Melati
     * @return String status of test
     */
    public String encodingTest(Melati melati) {
      Charset cs = Charset.forName(melati.getEncoding());
      CharsetEncoder e = cs.newEncoder();
      CharsetDecoder d = cs.newDecoder();
      if (e.canEncode(getChar())) {
        try {
          CharBuffer cb = CharBuffer.allocate(1).put(getChar());
          cb.flip();
          ByteBuffer bb = e.encode(cb);
          String result = d.decode(bb).toString();
          if (getChar().equals(result)) {
            return "Correct";
          } else {
            return "Wrong: '" + result;
          }
        }
        catch (CharacterCodingException cce) {
          return "Exception:" + cce.getMessage();
        }
      } else {
        return "Cannot";
      }
    }
    
  }

  /**
   * This data was originally copied from the unicode database..
   * <p>
   * <h2><i><a name="UCD_Terms"><br>UCD Terms of Use</a></i></h2>
   * <h3><i>Disclaimer</i></h3>
   * <blockquote>
   * <p><i>The Unicode Character Database is provided as is by Unicode, Inc. No 
   * claims are made as to fitness for any particular purpose. 
   * No warranties of any 
   * kind are expressed or implied. The recipient agrees to determine 
   * applicability 
   * of information provided. If this file has been purchased on magnetic or 
   * optical media from Unicode, Inc., the sole remedy for any claim will be 
   * exchange of defective media within 90 days of receipt.</i></p>
   * <p><i>This disclaimer is applicable for all other data files 
   * accompanying the 
   * Unicode Character Database, some of which have been compiled by the Unicode 
   * Consortium, and some of which have been supplied by other sources.</i></p>
   * </blockquote>
   * <h3><i>Limitations on Rights to Redistribute This Data</i></h3>
   * <blockquote>
   * <p><i>Recipient is granted the right to make copies in any form for 
   * internal 
   * distribution and to freely use the information supplied in the creation of 
   * products supporting the Unicode<sup>TM</sup> Standard. The files in the 
   * Unicode Character Database can be redistributed to third parties or other 
   * organizations (whether for profit or not) as long as this notice and the 
   * disclaimer notice are retained. 
   * Information can be extracted from these files 
   * and used in documentation or programs, as long as there is an accompanying 
   * notice indicating the source.</i></p>
   * </blockquote>
   *
   * <p><a href="http://www.unicode.org/unicode/copyright.html">
   * http://www.unicode.org/unicode/copyright.html</a>
   */
  public static final String[][] charData = {
    {"\u0040", "COMMERCIAL AT"},
    {"\u005E", "CIRCUMFLEX ACCENT"},
    {"\u0060", "GRAVE ACCENT"},
    {"\u00A1", "INVERTED EXCLAMATION MARK"},
    {"\u00A2", "CENT SIGN"},
    {"\u00A3", "POUND SIGN"},
    {"\u00A4", "CURRENCY SIGN"},
    {"\u00A5", "YEN SIGN"},
    {"\u00A6", "BROKEN BAR"},
    {"\u00A7", "SECTION SIGN"},
    {"\u00A8", "DIAERESIS"},
    {"\u00A9", "COPYRIGHT SIGN"},
    {"\u00AA", "FEMININE ORDINAL INDICATOR"},
    {"\u00AB", "LEFT-POINTING DOUBLE ANGLE QUOTATION  MARK"},
    {"\u00AC", "NOT SIGN"},
    {"\u00AD", "SOFT HYPHEN"},
    {"\u00AE", "REGISTERED SIGN"},
    {"\u00AF", "MACRON"},
    {"\u00B0", "DEGREE SIGN"},
    {"\u00B1", "PLUS-MINUS SIGN"},
    {"\u00B2", "SUPERSCRIPT TWO"},
    {"\u00B3", "SUPERSCRIPT THREE"},
    {"\u00B4", "ACUTE ACCENT"},
    {"\u00B5", "MICRO SIGN"},
    {"\u00B6", "PILCROW SIGN"},
    {"\u00B7", "MIDDLE DOT"},
    {"\u00B8", "CEDILLA"},
    {"\u00B9", "SUPERSCRIPT ONE"},
    {"\u00BA", "MASCULINE ORDINAL INDICATOR"},
    {"\u00BB", "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK"},
    {"\u00BC", "VULGAR FRACTION ONE QUARTER"},
    {"\u00BD", "VULGAR FRACTION ONE HALF"},
    {"\u00BE", "VULGAR FRACTION THREE QUARTERS"},
    {"\u00BF", "INVERTED QUESTION MARK"},
    {"\u00C0", "LATIN CAPITAL LETTER A WITH GRAVE"},
    {"\u00C1", "LATIN CAPITAL LETTER A WITH ACUTE"},
    {"\u00C2", "LATIN CAPITAL LETTER A WITH CIRCUMFLEX"},
    {"\u00C3", "LATIN CAPITAL LETTER A WITH TILDE"},
    {"\u00C4", "LATIN CAPITAL LETTER A WITH DIAERESIS"},
    {"\u00C5", "LATIN CAPITAL LETTER A WITH RING ABOVE"},
    {"\u00C6", "LATIN CAPITAL LETTER AE"},
    {"\u00C7", "LATIN CAPITAL LETTER C WITH CEDILLA"},
    {"\u00C8", "LATIN CAPITAL LETTER E WITH GRAVE"},
    {"\u00C9", "LATIN CAPITAL LETTER E WITH ACUTE"},
    {"\u00CA", "LATIN CAPITAL LETTER E WITH CIRCUMFLEX"},
    {"\u00CB", "LATIN CAPITAL LETTER E WITH DIAERESIS"},
    {"\u00CC", "LATIN CAPITAL LETTER I WITH GRAVE"},
    {"\u00CD", "LATIN CAPITAL LETTER I WITH ACUTE"},
    {"\u00CE", "LATIN CAPITAL LETTER I WITH CIRCUMFLEX"},
    {"\u00CF", "LATIN CAPITAL LETTER I WITH DIAERESIS"},
    {"\u00D0", "LATIN CAPITAL LETTER ETH"},
    {"\u00D1", "LATIN CAPITAL LETTER N WITH TILDE"},
    {"\u00D2", "LATIN CAPITAL LETTER O WITH GRAVE"},
    {"\u00D3", "LATIN CAPITAL LETTER O WITH ACUTE"},
    {"\u00D4", "LATIN CAPITAL LETTER O WITH CIRCUMFLEX"},
    {"\u00D5", "LATIN CAPITAL LETTER O WITH TILDE"},
    {"\u00D6", "LATIN CAPITAL LETTER O WITH DIAERESIS"},
    {"\u00D7", "MULTIPLICATION SIGN"},
    {"\u00D8", "LATIN CAPITAL LETTER O WITH STROKE"},
    {"\u00D9", "LATIN CAPITAL LETTER U WITH GRAVE"},
    {"\u00DA", "LATIN CAPITAL LETTER U WITH ACUTE"},
    {"\u00DB", "LATIN CAPITAL LETTER U WITH CIRCUMFLEX"},
    {"\u00DC", "LATIN CAPITAL LETTER U WITH DIAERESIS"},
    {"\u00DD", "LATIN CAPITAL LETTER Y WITH ACUTE"},
    {"\u00DE", "LATIN CAPITAL LETTER THORN"},
    {"\u00DF", "LATIN SMALL LETTER SHARP S"},
    {"\u00E0", "LATIN SMALL LETTER A WITH GRAVE"},
    {"\u00E1", "LATIN SMALL LETTER A WITH ACUTE"},
    {"\u00E2", "LATIN SMALL LETTER A WITH CIRCUMFLEX"},
    {"\u00E3", "LATIN SMALL LETTER A WITH TILDE"},
    {"\u00E4", "LATIN SMALL LETTER A WITH DIAERESIS"},
    {"\u00E5", "LATIN SMALL LETTER A WITH RING ABOVE"},
    {"\u00E6", "LATIN SMALL LETTER AE"},
    {"\u00E7", "LATIN SMALL LETTER C WITH CEDILLA"},
    {"\u00E8", "LATIN SMALL LETTER E WITH GRAVE"},
    {"\u00E9", "LATIN SMALL LETTER E WITH ACUTE"},
    {"\u00EA", "LATIN SMALL LETTER E WITH CIRCUMFLEX"},
    {"\u00EB", "LATIN SMALL LETTER E WITH DIAERESIS"},
    {"\u00EC", "LATIN SMALL LETTER I WITH GRAVE"},
    {"\u00ED", "LATIN SMALL LETTER I WITH ACUTE"},
    {"\u00EE", "LATIN SMALL LETTER I WITH CIRCUMFLEX"},
    {"\u00EF", "LATIN SMALL LETTER I WITH DIAERESIS"},
    {"\u00F0", "LATIN SMALL LETTER ETH"},
    {"\u00F1", "LATIN SMALL LETTER N WITH TILDE"},
    {"\u00F2", "LATIN SMALL LETTER O WITH GRAVE"},
    {"\u00F3", "LATIN SMALL LETTER O WITH ACUTE"},
    {"\u00F4", "LATIN SMALL LETTER O WITH CIRCUMFLEX"},
    {"\u00F5", "LATIN SMALL LETTER O WITH TILDE"},
    {"\u00F6", "LATIN SMALL LETTER O WITH DIAERESIS"},
    {"\u00F7", "DIVISION SIGN"},
    {"\u00F8", "LATIN SMALL LETTER O WITH STROKE"},
    {"\u00F9", "LATIN SMALL LETTER U WITH GRAVE"},
    {"\u00FA", "LATIN SMALL LETTER U WITH ACUTE"},
    {"\u00FB", "LATIN SMALL LETTER U WITH CIRCUMFLEX"},
    {"\u00FC", "LATIN SMALL LETTER U WITH DIAERESIS"},
    {"\u00FD", "LATIN SMALL LETTER Y WITH ACUTE"},
    {"\u00FE", "LATIN SMALL LETTER THORN"},
    {"\u00FF", "LATIN SMALL LETTER Y WITH DIAERESIS"},
    {"\u0100", "LATIN CAPITAL LETTER A WITH MACRON"},
    {"\u0101", "LATIN SMALL LETTER A WITH MACRON"},
    {"\u0102", "LATIN CAPITAL LETTER A WITH BREVE"},
    {"\u0103", "LATIN SMALL LETTER A WITH BREVE"},
    {"\u0104", "LATIN CAPITAL LETTER A WITH OGONEK"},
    {"\u0105", "LATIN SMALL LETTER A WITH OGONEK"},
    {"\u0106", "LATIN CAPITAL LETTER C WITH ACUTE"},
    {"\u0107", "LATIN SMALL LETTER C WITH ACUTE"},
    {"\u0108", "LATIN CAPITAL LETTER C WITH CIRCUMFLEX"},
    {"\u0109", "LATIN SMALL LETTER C WITH CIRCUMFLEX"},
    {"\u010A", "LATIN CAPITAL LETTER C WITH DOT ABOVE"},
    {"\u010B", "LATIN SMALL LETTER C WITH DOT ABOVE"},
    {"\u010C", "LATIN CAPITAL LETTER C WITH CARON"},
    {"\u010D", "LATIN SMALL LETTER C WITH CARON"},
    {"\u010E", "LATIN CAPITAL LETTER D WITH CARON"},
    {"\u010F", "LATIN SMALL LETTER D WITH CARON"},
    {"\u0110", "LATIN CAPITAL LETTER D WITH STROKE"},
    {"\u0111", "LATIN SMALL LETTER D WITH STROKE"},
    {"\u0112", "LATIN CAPITAL LETTER E WITH MACRON"},
    {"\u0113", "LATIN SMALL LETTER E WITH MACRON"},
    {"\u0114", "LATIN CAPITAL LETTER E WITH BREVE"},
    {"\u0115", "LATIN SMALL LETTER E WITH BREVE"},
    {"\u0116", "LATIN CAPITAL LETTER E WITH DOT ABOVE"},
    {"\u0117", "LATIN SMALL LETTER E WITH DOT ABOVE"},
    {"\u0118", "LATIN CAPITAL LETTER E WITH OGONEK"},
    {"\u0119", "LATIN SMALL LETTER E WITH OGONEK"},
    {"\u011A", "LATIN CAPITAL LETTER E WITH CARON"},
    {"\u011B", "LATIN SMALL LETTER E WITH CARON"},
    {"\u011C", "LATIN CAPITAL LETTER G WITH CIRCUMFLEX"},
    {"\u011D", "LATIN SMALL LETTER G WITH CIRCUMFLEX"},
    {"\u011E", "LATIN CAPITAL LETTER G WITH BREVE"},
    {"\u011F", "LATIN SMALL LETTER G WITH BREVE"},
    {"\u0120", "LATIN CAPITAL LETTER G WITH DOT ABOVE"},
    {"\u0121", "LATIN SMALL LETTER G WITH DOT ABOVE"},
    {"\u0122", "LATIN CAPITAL LETTER G WITH CEDILLA"},
    {"\u0123", "LATIN SMALL LETTER G WITH CEDILLA"},
    {"\u0124", "LATIN CAPITAL LETTER H WITH CIRCUMFLEX"},
    {"\u0125", "LATIN SMALL LETTER H WITH CIRCUMFLEX"},
    {"\u0126", "LATIN CAPITAL LETTER H WITH STROKE"},
    {"\u0127", "LATIN SMALL LETTER H WITH STROKE"},
    {"\u0128", "LATIN CAPITAL LETTER I WITH TILDE"},
    {"\u0129", "LATIN SMALL LETTER I WITH TILDE"},
    {"\u012A", "LATIN CAPITAL LETTER I WITH MACRON"},
    {"\u012B", "LATIN SMALL LETTER I WITH MACRON"},
    {"\u012C", "LATIN CAPITAL LETTER I WITH BREVE"},
    {"\u012D", "LATIN SMALL LETTER I WITH BREVE"},
    {"\u012E", "LATIN CAPITAL LETTER I WITH OGONEK"},
    {"\u012F", "LATIN SMALL LETTER I WITH OGONEK"},
    {"\u0130", "LATIN CAPITAL LETTER I WITH DOT ABOVE"},
    {"\u0131", "LATIN SMALL LETTER DOTLESS I"},
    {"\u0132", "LATIN CAPITAL LIGATURE IJ"},
    {"\u0133", "LATIN SMALL LIGATURE IJ"},
    {"\u0134", "LATIN CAPITAL LETTER J WITH CIRCUMFLEX"},
    {"\u0135", "LATIN SMALL LETTER J WITH CIRCUMFLEX"},
    {"\u0136", "LATIN CAPITAL LETTER K WITH CEDILLA"},
    {"\u0137", "LATIN SMALL LETTER K WITH CEDILLA"},
    {"\u0138", "LATIN SMALL LETTER KRA"},
    {"\u0139", "LATIN CAPITAL LETTER L WITH ACUTE"},
    {"\u013A", "LATIN SMALL LETTER L WITH ACUTE"},
    {"\u013B", "LATIN CAPITAL LETTER L WITH CEDILLA"},
    {"\u013C", "LATIN SMALL LETTER L WITH CEDILLA"},
    {"\u013D", "LATIN CAPITAL LETTER L WITH CARON"},
    {"\u013E", "LATIN SMALL LETTER L WITH CARON"},
    {"\u013F", "LATIN CAPITAL LETTER L WITH MIDDLE DOT"},
    {"\u0140", "LATIN SMALL LETTER L WITH MIDDLE DOT"},
    {"\u0141", "LATIN CAPITAL LETTER L WITH STROKE"},
    {"\u0142", "LATIN SMALL LETTER L WITH STROKE"},
    {"\u0143", "LATIN CAPITAL LETTER N WITH ACUTE"},
    {"\u0144", "LATIN SMALL LETTER N WITH ACUTE"},
    {"\u0145", "LATIN CAPITAL LETTER N WITH CEDILLA"},
    {"\u0146", "LATIN SMALL LETTER N WITH CEDILLA"},
    {"\u0147", "LATIN CAPITAL LETTER N WITH CARON"},
    {"\u0148", "LATIN SMALL LETTER N WITH CARON"},
    {"\u0149", "LATIN SMALL LETTER N PRECEDED BY APOSTROPHE"},
    {"\u014A", "LATIN CAPITAL LETTER ENG"},
    {"\u014B", "LATIN SMALL LETTER ENG"},
    {"\u014C", "LATIN CAPITAL LETTER O WITH MACRON"},
    {"\u014D", "LATIN SMALL LETTER O WITH MACRON"},
    {"\u014E", "LATIN CAPITAL LETTER O WITH BREVE"},
    {"\u014F", "LATIN SMALL LETTER O WITH BREVE"},
    {"\u0150", "LATIN CAPITAL LETTER O WITH DOUBLE ACUTE"},
    {"\u0151", "LATIN SMALL LETTER O WITH DOUBLE ACUTE"},
    {"\u0152", "LATIN CAPITAL LIGATURE OE"},
    {"\u0153", "LATIN SMALL LIGATURE OE"},
    {"\u0154", "LATIN CAPITAL LETTER R WITH ACUTE"},
    {"\u0155", "LATIN SMALL LETTER R WITH ACUTE"},
    {"\u0156", "LATIN CAPITAL LETTER R WITH CEDILLA"},
    {"\u0157", "LATIN SMALL LETTER R WITH CEDILLA"},
    {"\u0158", "LATIN CAPITAL LETTER R WITH CARON"},
    {"\u0159", "LATIN SMALL LETTER R WITH CARON"},
    {"\u015A", "LATIN CAPITAL LETTER S WITH ACUTE"},
    {"\u015B", "LATIN SMALL LETTER S WITH ACUTE"},
    {"\u015C", "LATIN CAPITAL LETTER S WITH CIRCUMFLEX"},
    {"\u015D", "LATIN SMALL LETTER S WITH CIRCUMFLEX"},
    {"\u015E", "LATIN CAPITAL LETTER S WITH CEDILLA"},
    {"\u015F", "LATIN SMALL LETTER S WITH CEDILLA"},
    {"\u0160", "LATIN CAPITAL LETTER S WITH CARON"},
    {"\u0161", "LATIN SMALL LETTER S WITH CARON"},
    {"\u0162", "LATIN CAPITAL LETTER T WITH CEDILLA"},
    {"\u0163", "LATIN SMALL LETTER T WITH CEDILLA"},
    {"\u0164", "LATIN CAPITAL LETTER T WITH CARON"},
    {"\u0165", "LATIN SMALL LETTER T WITH CARON"},
    {"\u0166", "LATIN CAPITAL LETTER T WITH STROKE"},
    {"\u0167", "LATIN SMALL LETTER T WITH STROKE"},
    {"\u0168", "LATIN CAPITAL LETTER U WITH TILDE"},
    {"\u0169", "LATIN SMALL LETTER U WITH TILDE"},
    {"\u016A", "LATIN CAPITAL LETTER U WITH MACRON"},
    {"\u016B", "LATIN SMALL LETTER U WITH MACRON"},
    {"\u016C", "LATIN CAPITAL LETTER U WITH BREVE"},
    {"\u016D", "LATIN SMALL LETTER U WITH BREVE"},
    {"\u016E", "LATIN CAPITAL LETTER U WITH RING ABOVE"},
    {"\u016F", "LATIN SMALL LETTER U WITH RING ABOVE"},
    {"\u0170", "LATIN CAPITAL LETTER U WITH DOUBLE ACUTE"},
    {"\u0171", "LATIN SMALL LETTER U WITH DOUBLE ACUTE"},
    {"\u0172", "LATIN CAPITAL LETTER U WITH OGONEK"},
    {"\u0173", "LATIN SMALL LETTER U WITH OGONEK"},
    {"\u0174", "LATIN CAPITAL LETTER W WITH CIRCUMFLEX"},
    {"\u0175", "LATIN SMALL LETTER W WITH CIRCUMFLEX"},
    {"\u0176", "LATIN CAPITAL LETTER Y WITH CIRCUMFLEX"},
    {"\u0177", "LATIN SMALL LETTER Y WITH CIRCUMFLEX"},
    {"\u0178", "LATIN CAPITAL LETTER Y WITH DIAERESIS"},
    {"\u0179", "LATIN CAPITAL LETTER Z WITH ACUTE"},
    {"\u017A", "LATIN SMALL LETTER Z WITH ACUTE"},
    {"\u017B", "LATIN CAPITAL LETTER Z WITH DOT ABOVE"},
    {"\u017C", "LATIN SMALL LETTER Z WITH DOT ABOVE"},
    {"\u017D", "LATIN CAPITAL LETTER Z WITH CARON"},
    {"\u017E", "LATIN SMALL LETTER Z WITH CARON"},
    {"\u017F", "LATIN SMALL LETTER LONG S"},
    {"\u0180", "LATIN SMALL LETTER B WITH STROKE"},
    {"\u0181", "LATIN CAPITAL LETTER B WITH HOOK"},
    {"\u0182", "LATIN CAPITAL LETTER B WITH TOPBAR"},
    {"\u0183", "LATIN SMALL LETTER B WITH TOPBAR"},
    {"\u0184", "LATIN CAPITAL LETTER TONE SIX"},
    {"\u0185", "LATIN SMALL LETTER TONE SIX"},
    {"\u0185", "LATIN SMALL LETTER TONE SIX"},
    {"\u0186", "LATIN CAPITAL LETTER OPEN O"},
    {"\u0187", "LATIN CAPITAL LETTER C WITH HOOK"},
    {"\u0188", "LATIN SMALL LETTER C WITH HOOK"},
    {"\u0189", "LATIN CAPITAL LETTER AFRICAN D"},
    {"\u018A", "LATIN CAPITAL LETTER D WITH HOOK"},
    {"\u018B", "LATIN CAPITAL LETTER D WITH TOPBAR"},
    {"\u018C", "LATIN SMALL LETTER D WITH TOPBAR"},
    {"\u018D", "LATIN SMALL LETTER TURNED DELTA"},
    {"\u018E", "LATIN CAPITAL LETTER REVERSED E"},
    {"\u018F", "LATIN CAPITAL LETTER SCHWA"},
    {"\u0190", "LATIN CAPITAL LETTER OPEN E"},
    {"\u0191", "LATIN CAPITAL LETTER F WITH HOOK"},
    {"\u0192", "LATIN SMALL LETTER F WITH HOOK"},
    {"\u0193", "LATIN CAPITAL LETTER G WITH HOOK"},
    {"\u0194", "LATIN CAPITAL LETTER GAMMA"},
    {"\u0195", "LATIN SMALL LETTER HV"},
    {"\u0196", "LATIN CAPITAL LETTER IOTA"},
    {"\u0197", "LATIN CAPITAL LETTER I WITH STROKE"},
    {"\u0198", "LATIN CAPITAL LETTER K WITH HOOK"},
    {"\u0199", "LATIN SMALL LETTER K WITH HOOK"},
    {"\u019A", "LATIN SMALL LETTER L WITH BAR"},
    {"\u019B", "LATIN SMALL LETTER LAMBDA WITH STROKE"},
    {"\u019C", "LATIN CAPITAL LETTER TURNED M"},
    {"\u019D", "LATIN CAPITAL LETTER N WITH LEFT HOOK"},
    {"\u019E", "LATIN SMALL LETTER N WITH LONG RIGHT LEG"},
    {"\u019F", "LATIN CAPITAL LETTER O WITH MIDDLE TILDE"},
    {"\u01A0", "LATIN CAPITAL LETTER O WITH HORN"},
    {"\u01A1", "LATIN SMALL LETTER O WITH HORN"},
    {"\u01A2", "LATIN CAPITAL LETTER OI"},
    {"\u01A3", "LATIN SMALL LETTER OI"},
    {"\u01A4", "LATIN CAPITAL LETTER P WITH HOOK"},
    {"\u01A5", "LATIN SMALL LETTER P WITH HOOK"},
    {"\u01A6", "LATIN LETTER YR"},
    {"\u01A7", "LATIN CAPITAL LETTER TONE TWO"},
    {"\u01A8", "LATIN SMALL LETTER TONE TWO"},
    {"\u01A9", "LATIN CAPITAL LETTER ESH"},
    {"\u01AA", "LATIN LETTER REVERSED ESH LOOP"},
    {"\u01AB", "LATIN SMALL LETTER T WITH PALATAL HOOK"},
    {"\u01AC", "LATIN CAPITAL LETTER T WITH HOOK"},
    {"\u01AD", "LATIN SMALL LETTER T WITH HOOK"},
    {"\u01AE", "LATIN CAPITAL LETTER T WITH RETROFLEX HOOK"},
    {"\u01AF", "LATIN CAPITAL LETTER U WITH HORN"},
    {"\u01B0", "LATIN SMALL LETTER U WITH HORN"},
    {"\u01B1", "LATIN CAPITAL LETTER UPSILON"},
    {"\u01B2", "LATIN CAPITAL LETTER V WITH HOOK"},
    {"\u01B3", "LATIN CAPITAL LETTER Y WITH HOOK"},
    {"\u01B4", "LATIN SMALL LETTER Y WITH HOOK"},
    {"\u01B5", "LATIN CAPITAL LETTER Z WITH STROKE"},
    {"\u01B6", "LATIN SMALL LETTER Z WITH STROKE"},
    {"\u01B7", "LATIN CAPITAL LETTER EZH"},
    {"\u01B8", "LATIN CAPITAL LETTER EZH REVERSED"},
    {"\u01B9", "LATIN SMALL LETTER EZH REVERSED"},
    {"\u01BA", "LATIN SMALL LETTER EZH WITH TAIL"},
    {"\u01BB", "LATIN LETTER TWO WITH STROKE"},
    {"\u01BC", "LATIN CAPITAL LETTER TONE FIVE"},
    {"\u01BD", "LATIN SMALL LETTER TONE FIVE"},
    {"\u01BE", "LATIN LETTER INVERTED GLOTTAL STOP WITH STROKE"},
    {"\u01BF", "LATIN LETTER WYNN"},
    {"\u01C0", "LATIN LETTER DENTAL CLICK"},
    {"\u01C1", "LATIN LETTER LATERAL CLICK"},
    {"\u01C2", "LATIN LETTER ALVEOLAR CLICK"},
    {"\u01C3", "LATIN LETTER RETROFLEX CLICK"},
    {"\u01C4", "LATIN CAPITAL LETTER DZ WITH CARON"},
    {"\u01C5", "LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON"},
    {"\u01C6", "LATIN SMALL LETTER DZ WITH CARON"},
    {"\u01C7", "LATIN CAPITAL LETTER LJ"},
    {"\u01C8", "LATIN CAPITAL LETTER L WITH SMALL LETTER J"},
    {"\u01C9", "LATIN SMALL LETTER LJ"},
    {"\u01CA", "LATIN CAPITAL LETTER NJ"},
    {"\u01CB", "LATIN CAPITAL LETTER N WITH SMALL LETTER J"},
    {"\u01CC", "LATIN SMALL LETTER NJ"},
    {"\u01CD", "LATIN CAPITAL LETTER A WITH CARON"},
    {"\u01CE", "LATIN SMALL LETTER A WITH CARON"},
    {"\u01CF", "LATIN CAPITAL LETTER I WITH CARON"},
    {"\u01D0", "LATIN SMALL LETTER I WITH CARON"},
    {"\u01D1", "LATIN CAPITAL LETTER O WITH CARON"},
    {"\u01D2", "LATIN SMALL LETTER O WITH CARON"},
    {"\u01D3", "LATIN CAPITAL LETTER U WITH CARON"},
    {"\u01D4", "LATIN SMALL LETTER U WITH CARON"},
    {"\u01D5", "LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON"},
    {"\u01D6", "LATIN SMALL LETTER U WITH DIAERESIS AND MACRON"},
    {"\u01D7", "LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE"},
    {"\u01D8", "LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE"},
    {"\u01D9", "LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON"},
    {"\u01DA", "LATIN SMALL LETTER U WITH DIAERESIS AND CARON"},
    {"\u01DB", "LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE"},
    {"\u01DC", "LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE"},
    {"\u01DD", "LATIN SMALL LETTER TURNED E"},
    {"\u01DE", "LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON"},
    {"\u01DF", "LATIN SMALL LETTER A WITH DIAERESIS AND MACRON"},
    {"\u01E0", "LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON"},
    {"\u01E1", "LATIN SMALL LETTER A WITH DOT ABOVE AND MACRON"},
    {"\u01E2", "LATIN CAPITAL LETTER AE WITH MACRON"},
    {"\u01E3", "LATIN SMALL LETTER AE WITH MACRON"},
    {"\u01E4", "LATIN CAPITAL LETTER G WITH STROKE"},
    {"\u01E5", "LATIN SMALL LETTER G WITH STROKE"},
    {"\u01E6", "LATIN CAPITAL LETTER G WITH CARON"},
    {"\u01E7", "LATIN SMALL LETTER G WITH CARON"},
    {"\u01E8", "LATIN CAPITAL LETTER K WITH CARON"},
    {"\u01E9", "LATIN SMALL LETTER K WITH CARON"},
    {"\u01EA", "LATIN CAPITAL LETTER O WITH OGONEK"},
    {"\u01EB", "LATIN SMALL LETTER O WITH OGONEK"},
    {"\u01EC", "LATIN CAPITAL LETTER O WITH OGONEK AND MACRON"},
    {"\u01ED", "LATIN SMALL LETTER O WITH OGONEK AND MACRON"},
    {"\u01EE", "LATIN CAPITAL LETTER EZH WITH CARON"},
    {"\u01EF", "LATIN SMALL LETTER EZH WITH CARON"},
    {"\u01F0", "LATIN SMALL LETTER J WITH CARON"},
    {"\u01F1", "LATIN CAPITAL LETTER DZ"},
    {"\u01F2", "LATIN CAPITAL LETTER D WITH SMALL LETTER Z"},
    {"\u01F3", "LATIN SMALL LETTER DZ"},
    {"\u01F4", "LATIN CAPITAL LETTER G WITH ACUTE"},
    {"\u01F5", "LATIN SMALL LETTER G WITH ACUTE"},
    {"\u01FA", "LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE"},
    {"\u01FB", "LATIN SMALL LETTER A WITH RING ABOVE AND ACUTE"},
    {"\u01FC", "LATIN CAPITAL LETTER AE WITH ACUTE"},
    {"\u01FD", "LATIN SMALL LETTER AE WITH ACUTE"},
    {"\u01FE", "LATIN CAPITAL LETTER O WITH STROKE AND ACUTE"},
    {"\u01FF", "LATIN SMALL LETTER O WITH STROKE AND ACUTE"},
    {"\u0200", "LATIN CAPITAL LETTER A WITH DOUBLE GRAVE"},
    {"\u0201", "LATIN SMALL LETTER A WITH DOUBLE GRAVE"},
    {"\u0202", "LATIN CAPITAL LETTER A WITH INVERTED BREVE"},
    {"\u0203", "LATIN SMALL LETTER A WITH INVERTED BREVE"},
    {"\u0204", "LATIN CAPITAL LETTER E WITH DOUBLE GRAVE"},
    {"\u0205", "LATIN SMALL LETTER E WITH DOUBLE GRAVE"},
    {"\u0206", "LATIN CAPITAL LETTER E WITH INVERTED BREVE"},
    {"\u0207", "LATIN SMALL LETTER E WITH INVERTED BREVE"},
    {"\u0208", "LATIN CAPITAL LETTER I WITH DOUBLE GRAVE"},
    {"\u0209", "LATIN SMALL LETTER I WITH DOUBLE GRAVE"},
    {"\u020A", "LATIN CAPITAL LETTER I WITH INVERTED BREVE"},
    {"\u020B", "LATIN SMALL LETTER I WITH INVERTED BREVE"},
    {"\u020C", "LATIN CAPITAL LETTER O WITH DOUBLE GRAVE"},
    {"\u020D", "LATIN SMALL LETTER O WITH DOUBLE GRAVE"},
    {"\u020E", "LATIN CAPITAL LETTER O WITH INVERTED BREVE"},
    {"\u020F", "LATIN SMALL LETTER O WITH INVERTED BREVE"},
    {"\u0210", "LATIN CAPITAL LETTER R WITH DOUBLE GRAVE"},
    {"\u0211", "LATIN SMALL LETTER R WITH DOUBLE GRAVE"},
    {"\u0212", "LATIN CAPITAL LETTER R WITH INVERTED BREVE"},
    {"\u0213", "LATIN SMALL LETTER R WITH INVERTED BREVE"},
    {"\u0214", "LATIN CAPITAL LETTER U WITH DOUBLE GRAVE"},
    {"\u0215", "LATIN SMALL LETTER U WITH DOUBLE GRAVE"},
    {"\u0216", "LATIN CAPITAL LETTER U WITH INVERTED BREVE"},
    {"\u0217", "LATIN SMALL LETTER U WITH INVERTED BREVE"},
    {"\u0250", "LATIN SMALL LETTER TURNED A"},
    {"\u0251", "LATIN SMALL LETTER ALPHA"},
    {"\u0252", "LATIN SMALL LETTER TURNED ALPHA"},
    {"\u0253", "LATIN SMALL LETTER B WITH HOOK"},
    {"\u0254", "LATIN SMALL LETTER OPEN O"},
    {"\u0255", "LATIN SMALL LETTER C WITH CURL"},
    {"\u0256", "LATIN SMALL LETTER D WITH TAIL"},
    {"\u0257", "LATIN SMALL LETTER D WITH HOOK"},
    {"\u0258", "LATIN SMALL LETTER REVERSED E"},
    {"\u0259", "LATIN SMALL LETTER SCHWA"},
    {"\u025A", "LATIN SMALL LETTER SCHWA WITH HOOK"},
    {"\u025B", "LATIN SMALL LETTER OPEN E"},
    {"\u025C", "LATIN SMALL LETTER REVERSED OPEN E"},
    {"\u025D", "LATIN SMALL LETTER REVERSED OPEN E WITH HOOK"},
    {"\u025E", "LATIN SMALL LETTER CLOSED REVERSED OPEN E"},
    {"\u025F", "LATIN SMALL LETTER DOTLESS J WITH STROKE"},
    {"\u0260", "LATIN SMALL LETTER G WITH HOOK"},
    {"\u0261", "LATIN SMALL LETTER SCRIPT G"},
    {"\u0262", "LATIN LETTER SMALL CAPITAL G"},
    {"\u0263", "LATIN SMALL LETTER GAMMA"},
    {"\u0264", "LATIN SMALL LETTER RAMS HORN"},
    {"\u0265", "LATIN SMALL LETTER TURNED H"},
    {"\u0266", "LATIN SMALL LETTER H WITH HOOK"},
    {"\u0267", "LATIN SMALL LETTER HENG WITH HOOK"},
    {"\u0268", "LATIN SMALL LETTER I WITH STROKE"},
    {"\u0269", "LATIN SMALL LETTER IOTA"},
    {"\u026A", "LATIN LETTER SMALL CAPITAL I"},
    {"\u026B", "LATIN SMALL LETTER L WITH MIDDLE TILDE"},
    {"\u026C", "LATIN SMALL LETTER L WITH BELT"},
    {"\u026D", "LATIN SMALL LETTER L WITH RETROFLEX HOOK"},
    {"\u026E", "LATIN SMALL LETTER LEZH"},
    {"\u026F", "LATIN SMALL LETTER TURNED M"},
    {"\u0270", "LATIN SMALL LETTER TURNED M WITH LONG LEG"},
    {"\u0271", "LATIN SMALL LETTER M WITH HOOK"},
    {"\u0272", "LATIN SMALL LETTER N WITH LEFT HOOK"},
    {"\u0273", "LATIN SMALL LETTER N WITH RETROFLEX HOOK"},
    {"\u0274", "LATIN LETTER SMALL CAPITAL N"},
    {"\u0275", "LATIN SMALL LETTER BARRED O"},
    {"\u0276", "LATIN LETTER SMALL CAPITAL OE"},
    {"\u0277", "LATIN SMALL LETTER CLOSED OMEGA"},
    {"\u0278", "LATIN SMALL LETTER PHI"},
    {"\u0279", "LATIN SMALL LETTER TURNED R"},
    {"\u027A", "LATIN SMALL LETTER TURNED R WITH LONG LEG"},
    {"\u027B", "LATIN SMALL LETTER TURNED R WITH HOOK"},
    {"\u027C", "LATIN SMALL LETTER R WITH LONG LEG"},
    {"\u027D", "LATIN SMALL LETTER R WITH TAIL"},
    {"\u027E", "LATIN SMALL LETTER R WITH FISHHOOK"},
    {"\u027F", "LATIN SMALL LETTER REVERSED R WITH FISHHOOK"},
    {"\u0280", "LATIN LETTER SMALL CAPITAL R"},
    {"\u0281", "LATIN LETTER SMALL CAPITAL INVERTED R"},
    {"\u0282", "LATIN SMALL LETTER S WITH HOOK"},
    {"\u0283", "LATIN SMALL LETTER ESH"},
    {"\u0284", "LATIN SMALL LETTER DOTLESS J WITH STROKE AND HOOK"},
    {"\u0285", "LATIN SMALL LETTER SQUAT REVERSED ESH"},
    {"\u0286", "LATIN SMALL LETTER ESH WITH CURL"},
    {"\u0287", "LATIN SMALL LETTER TURNED T"},
    {"\u0288", "LATIN SMALL LETTER T WITH RETROFLEX HOOK"},
    {"\u0289", "LATIN SMALL LETTER U BAR"},
    {"\u028A", "LATIN SMALL LETTER UPSILON"},
    {"\u028B", "LATIN SMALL LETTER V WITH HOOK"},
    {"\u028C", "LATIN SMALL LETTER TURNED V"},
    {"\u028D", "LATIN SMALL LETTER TURNED W"},
    {"\u028E", "LATIN SMALL LETTER TURNED Y"},
    {"\u028F", "LATIN LETTER SMALL CAPITAL Y"},
    {"\u0290", "LATIN SMALL LETTER Z WITH RETROFLEX HOOK"},
    {"\u0291", "LATIN SMALL LETTER Z WITH CURL"},
    {"\u0292", "LATIN SMALL LETTER EZH"},
    {"\u0293", "LATIN SMALL LETTER EZH WITH CURL"},
    {"\u0294", "LATIN LETTER GLOTTAL STOP"},
    {"\u0295", "LATIN LETTER PHARYNGEAL VOICED FRICATIVE"},
    {"\u0296", "LATIN LETTER INVERTED GLOTTAL STOP"},
    {"\u0297", "LATIN LETTER STRETCHED C"},
    {"\u0298", "LATIN LETTER BILABIAL CLICK"},
    {"\u0299", "LATIN LETTER SMALL CAPITAL B"},
    {"\u029A", "LATIN SMALL LETTER CLOSED OPEN E"},
    {"\u029B", "LATIN LETTER SMALL CAPITAL G WITH HOOK"},
    {"\u029C", "LATIN LETTER SMALL CAPITAL H"},
    {"\u029D", "LATIN SMALL LETTER J WITH CROSSED-TAIL"},
    {"\u029E", "LATIN SMALL LETTER TURNED K"},
    {"\u029F", "LATIN LETTER SMALL CAPITAL L"},
    {"\u02A0", "LATIN SMALL LETTER Q WITH HOOK"},
    {"\u02A1", "LATIN LETTER GLOTTAL STOP WITH STROKE"},
    {"\u02A2", "LATIN LETTER REVERSED GLOTTAL STOP WITH STROKE"},
    {"\u02A3", "LATIN SMALL LETTER DZ DIGRAPH"},
    {"\u02A4", "LATIN SMALL LETTER DEZH DIGRAPH"},
    {"\u02A5", "LATIN SMALL LETTER DZ DIGRAPH WITH CURL"},
    {"\u02A6", "LATIN SMALL LETTER TS DIGRAPH"},
    {"\u02A7", "LATIN SMALL LETTER TESH DIGRAPH"},
    {"\u02A8", "LATIN SMALL LETTER TC DIGRAPH WITH CURL"},
    {"\u02B0", "MODIFIER LETTER SMALL H"},
    {"\u02B1", "MODIFIER LETTER SMALL H WITH HOOK"},
    {"\u02B2", "MODIFIER LETTER SMALL J"},
    {"\u02B3", "MODIFIER LETTER SMALL R"},
    {"\u02B4", "MODIFIER LETTER SMALL TURNED R"},
    {"\u02B5", "MODIFIER LETTER SMALL TURNED R WITH HOOK"},
    {"\u02B6", "MODIFIER LETTER SMALL CAPITAL INVERTED R"},
    {"\u02B7", "MODIFIER LETTER SMALL W"},
    {"\u02B8", "MODIFIER LETTER SMALL Y"},
    {"\u02B9", "MODIFIER LETTER PRIME"},
    {"\u02BA", "MODIFIER LETTER DOUBLE PRIME"},
    {"\u02BB", "MODIFIER LETTER TURNED COMMA"},
    {"\u02BC", "MODIFIER LETTER APOSTROPHE"},
    {"\u02BD", "MODIFIER LETTER REVERSED COMMA"},
    {"\u02BE", "MODIFIER LETTER RIGHT HALF RING"},
    {"\u02BF", "MODIFIER LETTER LEFT HALF RING"},
    {"\u02C0", "MODIFIER LETTER GLOTTAL STOP"},
    {"\u02C1", "MODIFIER LETTER REVERSED GLOTTAL STOP"},
    {"\u02C2", "MODIFIER LETTER LEFT ARROWHEAD"},
    {"\u02C3", "MODIFIER LETTER RIGHT ARROWHEAD"},
    {"\u02C4", "MODIFIER LETTER UP ARROWHEAD"},
    {"\u02C5", "MODIFIER LETTER DOWN ARROWHEAD"},
    {"\u02C6", "MODIFIER LETTER CIRCUMFLEX ACCENT"},
    {"\u02C7", "CARON"},
    {"\u02C8", "MODIFIER LETTER VERTICAL LINE"},
    {"\u02C9", "MODIFIER LETTER MACRON"},
    {"\u02CA", "MODIFIER LETTER ACUTE ACCENT"},
    {"\u02CB", "MODIFIER LETTER GRAVE ACCENT"},
    {"\u02CC", "MODIFIER LETTER LOW VERTICAL LINE"},
    {"\u02CD", "MODIFIER LETTER LOW MACRON"},
    {"\u02CE", "MODIFIER LETTER LOW GRAVE ACCENT"},
    {"\u02CF", "MODIFIER LETTER LOW ACUTE ACCENT"},
    {"\u02D0", "MODIFIER LETTER TRIANGULAR COLON"},
    {"\u02D1", "MODIFIER LETTER HALF TRIANGULAR COLON"},
    {"\u02D2", "MODIFIER LETTER CENTRED RIGHT HALF RING"},
    {"\u02D3", "MODIFIER LETTER CENTRED LEFT HALF RING"},
    {"\u02D4", "MODIFIER LETTER UP TACK"},
    {"\u02D5", "MODIFIER LETTER DOWN TACK"},
    {"\u02D6", "MODIFIER LETTER PLUS SIGN"},
    {"\u02D7", "MODIFIER LETTER MINUS SIGN"},
    {"\u02D8", "BREVE"},
    {"\u02D9", "DOT ABOVE"},
    {"\u02DA", "RING ABOVE"},
    {"\u02DB", "OGONEK"},
    {"\u02DC", "SMALL TILDE"},
    {"\u02DD", "DOUBLE ACUTE ACCENT"},
    {"\u02DE", "MODIFIER LETTER RHOTIC HOOK"},
    {"\u02E0", "MODIFIER LETTER SMALL GAMMA"},
    {"\u02E1", "MODIFIER LETTER SMALL L"},
    {"\u02E2", "MODIFIER LETTER SMALL S"},
    {"\u02E3", "MODIFIER LETTER SMALL X"},
    {"\u02E4", "MODIFIER LETTER SMALL REVERSED GLOTTAL STOP"},
    {"\u02E5", "MODIFIER LETTER EXTRA-HIGH TONE BAR"},
    {"\u02E6", "MODIFIER LETTER HIGH TONE BAR"},
    {"\u02E7", "MODIFIER LETTER MID TONE BAR"},
    {"\u02E8", "MODIFIER LETTER LOW TONE BAR"},
    {"\u02E9", "MODIFIER LETTER EXTRA-LOW TONE BAR"},
    {"\u0300", "COMBINING GRAVE ACCENT"},
    {"\u0301", "COMBINING ACUTE ACCENT"},
    {"\u0302", "COMBINING CIRCUMFLEX ACCENT"},
    {"\u0303", "COMBINING TILDE"},
    {"\u0304", "COMBINING MACRON"},
    {"\u0305", "COMBINING OVERLINE"},
    {"\u0306", "COMBINING BREVE"},
    {"\u0307", "COMBINING DOT ABOVE"},
    {"\u0308", "COMBINING DIAERESIS"},
    {"\u0309", "COMBINING HOOK ABOVE"},
    {"\u030A", "COMBINING RING ABOVE"},
    {"\u030B", "COMBINING DOUBLE ACUTE ACCENT"},
    {"\u030C", "COMBINING CARON"},
    {"\u030D", "COMBINING VERTICAL LINE ABOVE"},
    {"\u030E", "COMBINING DOUBLE VERTICAL LINE ABOVE"},
    {"\u030F", "COMBINING DOUBLE GRAVE ACCENT"},
    {"\u0310", "COMBINING CANDRABINDU"},
    {"\u0311", "COMBINING INVERTED BREVE"},
    {"\u0312", "COMBINING TURNED COMMA ABOVE"},
    {"\u0313", "COMBINING COMMA ABOVE"},
    {"\u0314", "COMBINING REVERSED COMMA ABOVE"},
    {"\u0315", "COMBINING COMMA ABOVE RIGHT"},
    {"\u0316", "COMBINING GRAVE ACCENT BELOW"},
    {"\u0317", "COMBINING ACUTE ACCENT BELOW"},
    {"\u0318", "COMBINING LEFT TACK BELOW"},
    {"\u0319", "COMBINING RIGHT TACK BELOW"},
    {"\u031A", "COMBINING LEFT ANGLE ABOVE"},
    {"\u031B", "COMBINING HORN"},
    {"\u031C", "COMBINING LEFT HALF RING BELOW"},
    {"\u031D", "COMBINING UP TACK BELOW"},
    {"\u031E", "COMBINING DOWN TACK BELOW"},
    {"\u031F", "COMBINING PLUS SIGN BELOW"},
    {"\u0320", "COMBINING MINUS SIGN BELOW"},
    {"\u0321", "COMBINING PALATALIZED HOOK BELOW"},
    {"\u0322", "COMBINING RETROFLEX HOOK BELOW"},
    {"\u0323", "COMBINING DOT BELOW"},
    {"\u0324", "COMBINING DIAERESIS BELOW"},
    {"\u0325", "COMBINING RING BELOW"},
    {"\u0326", "COMBINING COMMA BELOW"},
    {"\u0327", "COMBINING CEDILLA"},
    {"\u0328", "COMBINING OGONEK"},
    {"\u0329", "COMBINING VERTICAL LINE BELOW"},
    {"\u032A", "COMBINING BRIDGE BELOW"},
    {"\u032B", "COMBINING INVERTED DOUBLE ARCH BELOW"},
    {"\u032C", "COMBINING CARON BELOW"},
    {"\u032D", "COMBINING CIRCUMFLEX ACCENT BELOW"},
    {"\u032E", "COMBINING BREVE BELOW"},
    {"\u032F", "COMBINING INVERTED BREVE BELOW"},
    {"\u0330", "COMBINING TILDE BELOW"},
    {"\u0331", "COMBINING MACRON BELOW"},
    {"\u0332", "COMBINING LOW LINE"},
    {"\u0333", "COMBINING DOUBLE LOW LINE"},
    {"\u0334", "COMBINING TILDE OVERLAY"},
    {"\u0335", "COMBINING SHORT STROKE OVERLAY"},
    {"\u0336", "COMBINING LONG STROKE OVERLAY"},
    {"\u0337", "COMBINING SHORT SOLIDUS OVERLAY"},
    {"\u0338", "COMBINING LONG SOLIDUS OVERLAY"},
    {"\u0339", "COMBINING RIGHT HALF RING BELOW"},
    {"\u033A", "COMBINING INVERTED BRIDGE BELOW"},
    {"\u033B", "COMBINING SQUARE BELOW"},
    {"\u033C", "COMBINING SEAGULL BELOW"},
    {"\u033D", "COMBINING X ABOVE"},
    {"\u033E", "COMBINING VERTICAL TILDE"},
    {"\u033F", "COMBINING DOUBLE OVERLINE"},
    {"\u0340", "COMBINING GRAVE TONE MARK"},
    {"\u0341", "COMBINING ACUTE TONE MARK"},
    {"\u0342", "COMBINING GREEK PERISPOMENI"},
    {"\u0343", "COMBINING GREEK KORONIS"},
    {"\u0344", "COMBINING GREEK DIALYTIKA TONOS"},
    {"\u0345", "COMBINING GREEK YPOGEGRAMMENI"},
    {"\u0360", "COMBINING DOUBLE TILDE"},
    {"\u0361", "COMBINING DOUBLE INVERTED BREVE"},
    {"\u0374", "GREEK NUMERAL SIGN"},
    {"\u0375", "GREEK LOWER NUMERAL SIGN"},
    {"\u037A", "GREEK YPOGEGRAMMENI"},
    {"\u037E", "GREEK QUESTION MARK"},
    {"\u0384", "GREEK TONOS"},
    {"\u0385", "GREEK DIALYTIKA TONOS"},
    {"\u0386", "GREEK CAPITAL LETTER ALPHA WITH TONOS"},
    {"\u0387", "GREEK ANO TELEIA"},
    {"\u0388", "GREEK CAPITAL LETTER EPSILON WITH TONOS"},
    {"\u0389", "GREEK CAPITAL LETTER ETA WITH TONOS"},
    {"\u038A", "GREEK CAPITAL LETTER IOTA WITH TONOS"},
    {"\u038C", "GREEK CAPITAL LETTER OMICRON WITH TONOS"},
    {"\u038E", "GREEK CAPITAL LETTER UPSILON WITH TONOS"},
    {"\u038F", "GREEK CAPITAL LETTER OMEGA WITH TONOS"},
    {"\u0390", "GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS"},
    {"\u0391", "GREEK CAPITAL LETTER ALPHA"},
    {"\u0392", "GREEK CAPITAL LETTER BETA"},
    {"\u0393", "GREEK CAPITAL LETTER GAMMA"},
    {"\u0394", "GREEK CAPITAL LETTER DELTA"},
    {"\u0395", "GREEK CAPITAL LETTER EPSILON"},
    {"\u0396", "GREEK CAPITAL LETTER ZETA"},
    {"\u0397", "GREEK CAPITAL LETTER ETA"},
    {"\u0398", "GREEK CAPITAL LETTER THETA"},
    {"\u0399", "GREEK CAPITAL LETTER IOTA"},
    {"\u039A", "GREEK CAPITAL LETTER KAPPA"},
    {"\u039B", "GREEK CAPITAL LETTER LAMDA"},
    {"\u039C", "GREEK CAPITAL LETTER MU"},
    {"\u039D", "GREEK CAPITAL LETTER NU"},
    {"\u039E", "GREEK CAPITAL LETTER XI"},
    {"\u039F", "GREEK CAPITAL LETTER OMICRON"},
    {"\u03A0", "GREEK CAPITAL LETTER PI"},
    {"\u03A1", "GREEK CAPITAL LETTER RHO"},
    {"\u03A3", "GREEK CAPITAL LETTER SIGMA"},
    {"\u03A4", "GREEK CAPITAL LETTER TAU"},
    {"\u03A5", "GREEK CAPITAL LETTER UPSILON"},
    {"\u03A6", "GREEK CAPITAL LETTER PHI"},
    {"\u03A7", "GREEK CAPITAL LETTER CHI"},
    {"\u03A8", "GREEK CAPITAL LETTER PSI"},
    {"\u03A9", "GREEK CAPITAL LETTER OMEGA"},
    {"\u03AA", "GREEK CAPITAL LETTER IOTA WITH DIALYTIKA"},
    {"\u03AB", "GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA"},
    {"\u03AC", "GREEK SMALL LETTER ALPHA WITH TONOS"},
    {"\u03AD", "GREEK SMALL LETTER EPSILON WITH TONOS"},
    {"\u03AE", "GREEK SMALL LETTER ETA WITH TONOS"},
    {"\u03AF", "GREEK SMALL LETTER IOTA WITH TONOS"},
    {"\u03B0", "GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS"},
    {"\u03B1", "GREEK SMALL LETTER ALPHA"},
    {"\u03B2", "GREEK SMALL LETTER BETA"},
    {"\u03B3", "GREEK SMALL LETTER GAMMA"},
    {"\u03B4", "GREEK SMALL LETTER DELTA"},
    {"\u03B5", "GREEK SMALL LETTER EPSILON"},
    {"\u03B6", "GREEK SMALL LETTER ZETA"},
    {"\u03B7", "GREEK SMALL LETTER ETA"},
    {"\u03B8", "GREEK SMALL LETTER THETA"},
    {"\u03B9", "GREEK SMALL LETTER IOTA"},
    {"\u03BA", "GREEK SMALL LETTER KAPPA"},
    {"\u03BB", "GREEK SMALL LETTER LAMDA"},
    {"\u03BC", "GREEK SMALL LETTER MU"},
    {"\u03BD", "GREEK SMALL LETTER NU"},
    {"\u03BE", "GREEK SMALL LETTER XI"},
    {"\u03BF", "GREEK SMALL LETTER OMICRON"},
    {"\u03C0", "GREEK SMALL LETTER PI"},
    {"\u03C1", "GREEK SMALL LETTER RHO"},
    {"\u03C2", "GREEK SMALL LETTER FINAL SIGMA"},
    {"\u03C3", "GREEK SMALL LETTER SIGMA"},
    {"\u03C4", "GREEK SMALL LETTER TAU"},
    {"\u03C5", "GREEK SMALL LETTER UPSILON"},
    {"\u03C6", "GREEK SMALL LETTER PHI"},
    {"\u03C7", "GREEK SMALL LETTER CHI"},
    {"\u03C8", "GREEK SMALL LETTER PSI"},
    {"\u03C9", "GREEK SMALL LETTER OMEGA"},
    {"\u03CA", "GREEK SMALL LETTER IOTA WITH DIALYTIKA"},
    {"\u03CB", "GREEK SMALL LETTER UPSILON WITH DIALYTIKA"},
    {"\u03CC", "GREEK SMALL LETTER OMICRON WITH TONOS"},
    {"\u03CD", "GREEK SMALL LETTER UPSILON WITH TONOS"},
    {"\u03CE", "GREEK SMALL LETTER OMEGA WITH TONOS"},
    {"\u03D0", "GREEK BETA SYMBOL"},
    {"\u03D1", "GREEK THETA SYMBOL"},
    {"\u03D2", "GREEK UPSILON WITH HOOK SYMBOL"},
    {"\u03D3", "GREEK UPSILON WITH ACUTE AND HOOK SYMBOL"},
    {"\u03D4", "GREEK UPSILON WITH DIAERESIS AND HOOK SYMBOL"},
    {"\u03D5", "GREEK PHI SYMBOL"},
    {"\u03D6", "GREEK PI SYMBOL"},
    {"\u03DA", "GREEK LETTER STIGMA"},
    {"\u03DC", "GREEK LETTER DIGAMMA"},
    {"\u03DE", "GREEK LETTER KOPPA"},
    {"\u03E0", "GREEK LETTER SAMPI"},
    {"\u03E2", "COPTIC CAPITAL LETTER SHEI"},
    {"\u03E3", "COPTIC SMALL LETTER SHEI"},
    {"\u03E4", "COPTIC CAPITAL LETTER FEI"},
    {"\u03E5", "COPTIC SMALL LETTER FEI"},
    {"\u03E6", "COPTIC CAPITAL LETTER KHEI"},
    {"\u03E7", "COPTIC SMALL LETTER KHEI"},
    {"\u03E8", "COPTIC CAPITAL LETTER HORI"},
    {"\u03E9", "COPTIC SMALL LETTER HORI"},
    {"\u03EA", "COPTIC CAPITAL LETTER GANGIA"},
    {"\u03EB", "COPTIC SMALL LETTER GANGIA"},
    {"\u03EC", "COPTIC CAPITAL LETTER SHIMA"},
    {"\u03ED", "COPTIC SMALL LETTER SHIMA"},
    {"\u03EE", "COPTIC CAPITAL LETTER DEI"},
    {"\u03EF", "COPTIC SMALL LETTER DEI"},
    {"\u03F0", "GREEK KAPPA SYMBOL"},
    {"\u03F1", "GREEK RHO SYMBOL"},
    {"\u03F2", "GREEK LUNATE SIGMA SYMBOL"},
    {"\u03F3", "GREEK LETTER YOT"},
    {"\u0401", "CYRILLIC CAPITAL LETTER IO"},
    {"\u0402", "CYRILLIC CAPITAL LETTER DJE"},
    {"\u0403", "CYRILLIC CAPITAL LETTER GJE"},
    {"\u0404", "CYRILLIC CAPITAL LETTER UKRAINIAN IE"},
    {"\u0405", "CYRILLIC CAPITAL LETTER DZE"},
    {"\u0406", "CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I"},
    {"\u0407", "CYRILLIC CAPITAL LETTER YI"},
    {"\u0408", "CYRILLIC CAPITAL LETTER JE"},
    {"\u0409", "CYRILLIC CAPITAL LETTER LJE"},
    {"\u040A", "CYRILLIC CAPITAL LETTER NJE"},
    {"\u040B", "CYRILLIC CAPITAL LETTER TSHE"},
    {"\u040C", "CYRILLIC CAPITAL LETTER KJE"},
    {"\u040E", "CYRILLIC CAPITAL LETTER SHORT U"},
    {"\u040F", "CYRILLIC CAPITAL LETTER DZHE"},
    {"\u0410", "CYRILLIC CAPITAL LETTER A"},
    {"\u0411", "CYRILLIC CAPITAL LETTER BE"},
    {"\u0412", "CYRILLIC CAPITAL LETTER VE"},
    {"\u0413", "CYRILLIC CAPITAL LETTER GHE"},
    {"\u0414", "CYRILLIC CAPITAL LETTER DE"},
    {"\u0415", "CYRILLIC CAPITAL LETTER IE"},
    {"\u0416", "CYRILLIC CAPITAL LETTER ZHE"},
    {"\u0417", "CYRILLIC CAPITAL LETTER ZE"},
    {"\u0418", "CYRILLIC CAPITAL LETTER I"},
    {"\u0419", "CYRILLIC CAPITAL LETTER SHORT I"},
    {"\u041A", "CYRILLIC CAPITAL LETTER KA"},
    {"\u041B", "CYRILLIC CAPITAL LETTER EL"},
    {"\u041C", "CYRILLIC CAPITAL LETTER EM"},
    {"\u041D", "CYRILLIC CAPITAL LETTER EN"},
    {"\u041E", "CYRILLIC CAPITAL LETTER O"},
    {"\u041F", "CYRILLIC CAPITAL LETTER PE"},
    {"\u0420", "CYRILLIC CAPITAL LETTER ER"},
    {"\u0421", "CYRILLIC CAPITAL LETTER ES"},
    {"\u0422", "CYRILLIC CAPITAL LETTER TE"},
    {"\u0423", "CYRILLIC CAPITAL LETTER U"},
    {"\u0424", "CYRILLIC CAPITAL LETTER EF"},
    {"\u0425", "CYRILLIC CAPITAL LETTER HA"},
    {"\u0426", "CYRILLIC CAPITAL LETTER TSE"},
    {"\u0427", "CYRILLIC CAPITAL LETTER CHE"},
    {"\u0428", "CYRILLIC CAPITAL LETTER SHA"},
    {"\u0429", "CYRILLIC CAPITAL LETTER SHCHA"},
    {"\u042A", "CYRILLIC CAPITAL LETTER HARD SIGN"},
    {"\u042B", "CYRILLIC CAPITAL LETTER YERU"},
    {"\u042C", "CYRILLIC CAPITAL LETTER SOFT SIGN"},
    {"\u042D", "CYRILLIC CAPITAL LETTER E"},
    {"\u042E", "CYRILLIC CAPITAL LETTER YU"},
    {"\u042F", "CYRILLIC CAPITAL LETTER YA"},
    {"\u0430", "CYRILLIC SMALL LETTER A"},
    {"\u0431", "CYRILLIC SMALL LETTER BE"},
    {"\u0432", "CYRILLIC SMALL LETTER VE"},
    {"\u0433", "CYRILLIC SMALL LETTER GHE"},
    {"\u0434", "CYRILLIC SMALL LETTER DE"},
    {"\u0435", "CYRILLIC SMALL LETTER IE"},
    {"\u0436", "CYRILLIC SMALL LETTER ZHE"},
    {"\u0437", "CYRILLIC SMALL LETTER ZE"},
    {"\u0438", "CYRILLIC SMALL LETTER I"},
    {"\u0439", "CYRILLIC SMALL LETTER SHORT I"},
    {"\u043A", "CYRILLIC SMALL LETTER KA"},
    {"\u043B", "CYRILLIC SMALL LETTER EL"},
    {"\u043C", "CYRILLIC SMALL LETTER EM"},
    {"\u043D", "CYRILLIC SMALL LETTER EN"},
    {"\u043E", "CYRILLIC SMALL LETTER O"},
    {"\u043F", "CYRILLIC SMALL LETTER PE"},
    {"\u0440", "CYRILLIC SMALL LETTER ER"},
    {"\u0441", "CYRILLIC SMALL LETTER ES"},
    {"\u0442", "CYRILLIC SMALL LETTER TE"},
    {"\u0443", "CYRILLIC SMALL LETTER U"},
    {"\u0444", "CYRILLIC SMALL LETTER EF"},
    {"\u0445", "CYRILLIC SMALL LETTER HA"},
    {"\u0446", "CYRILLIC SMALL LETTER TSE"},
    {"\u0447", "CYRILLIC SMALL LETTER CHE"},
    {"\u0448", "CYRILLIC SMALL LETTER SHA"},
    {"\u0449", "CYRILLIC SMALL LETTER SHCHA"},
    {"\u044A", "CYRILLIC SMALL LETTER HARD SIGN"},
    {"\u044B", "CYRILLIC SMALL LETTER YERU"},
    {"\u044C", "CYRILLIC SMALL LETTER SOFT SIGN"},
    {"\u044D", "CYRILLIC SMALL LETTER E"},
    {"\u044E", "CYRILLIC SMALL LETTER YU"},
    {"\u044F", "CYRILLIC SMALL LETTER YA"},
    {"\u0451", "CYRILLIC SMALL LETTER IO"},
    {"\u0452", "CYRILLIC SMALL LETTER DJE"},
    {"\u0453", "CYRILLIC SMALL LETTER GJE"},
    {"\u0454", "CYRILLIC SMALL LETTER UKRAINIAN IE"},
    {"\u0455", "CYRILLIC SMALL LETTER DZE"},
    {"\u0456", "CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I"},
    {"\u0457", "CYRILLIC SMALL LETTER YI"},
    {"\u0458", "CYRILLIC SMALL LETTER JE"},
    {"\u0459", "CYRILLIC SMALL LETTER LJE"},
    {"\u045A", "CYRILLIC SMALL LETTER NJE"},
    {"\u045B", "CYRILLIC SMALL LETTER TSHE"},
    {"\u045C", "CYRILLIC SMALL LETTER KJE"},
    {"\u045E", "CYRILLIC SMALL LETTER SHORT U"},
    {"\u045F", "CYRILLIC SMALL LETTER DZHE"},
    {"\u0460", "CYRILLIC CAPITAL LETTER OMEGA"},
    {"\u0461", "CYRILLIC SMALL LETTER OMEGA"},
    {"\u0462", "CYRILLIC CAPITAL LETTER YAT"},
    {"\u0463", "CYRILLIC SMALL LETTER YAT"},
    {"\u0464", "CYRILLIC CAPITAL LETTER IOTIFIED E"},
    {"\u0465", "CYRILLIC SMALL LETTER IOTIFIED E"},
    {"\u0466", "CYRILLIC CAPITAL LETTER LITTLE YUS"},
    {"\u0467", "CYRILLIC SMALL LETTER LITTLE YUS"},
    {"\u0468", "CYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUS"},
    {"\u0469", "CYRILLIC SMALL LETTER IOTIFIED LITTLE YUS"},
    {"\u046A", "CYRILLIC CAPITAL LETTER BIG YUS"},
    {"\u046B", "CYRILLIC SMALL LETTER BIG YUS"},
    {"\u046C", "CYRILLIC CAPITAL LETTER IOTIFIED BIG YUS"},
    {"\u046D", "CYRILLIC SMALL LETTER IOTIFIED BIG YUS"},
    {"\u046E", "CYRILLIC CAPITAL LETTER KSI"},
    {"\u046F", "CYRILLIC SMALL LETTER KSI"},
    {"\u0470", "CYRILLIC CAPITAL LETTER PSI"},
    {"\u0471", "CYRILLIC SMALL LETTER PSI"},
    {"\u0472", "CYRILLIC CAPITAL LETTER FITA"},
    {"\u0473", "CYRILLIC SMALL LETTER FITA"},
    {"\u0474", "CYRILLIC CAPITAL LETTER IZHITSA"},
    {"\u0475", "CYRILLIC SMALL LETTER IZHITSA"},
    {"\u0476", "CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT"},
    {"\u0477", "CYRILLIC SMALL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT"},
    {"\u0478", "CYRILLIC CAPITAL LETTER UK"},
    {"\u0479", "CYRILLIC SMALL LETTER UK"},
    {"\u047A", "CYRILLIC CAPITAL LETTER ROUND OMEGA"},
    {"\u047B", "CYRILLIC SMALL LETTER ROUND OMEGA"},
    {"\u047C", "CYRILLIC CAPITAL LETTER OMEGA WITH TITLO"},
    {"\u047D", "CYRILLIC SMALL LETTER OMEGA WITH TITLO"},
    {"\u047E", "CYRILLIC CAPITAL LETTER OT"},
    {"\u047F", "CYRILLIC SMALL LETTER OT"},
    {"\u0480", "CYRILLIC CAPITAL LETTER KOPPA"},
    {"\u0481", "CYRILLIC SMALL LETTER KOPPA"},
    {"\u0482", "CYRILLIC THOUSANDS SIGN"},
    {"\u0483", "COMBINING CYRILLIC TITLO"},
    {"\u0484", "COMBINING CYRILLIC PALATALIZATION"},
    {"\u0485", "COMBINING CYRILLIC DASIA PNEUMATA"},
    {"\u0486", "COMBINING CYRILLIC PSILI PNEUMATA"},
    {"\u0490", "CYRILLIC CAPITAL LETTER GHE WITH UPTURN"},
    {"\u0491", "CYRILLIC SMALL LETTER GHE WITH UPTURN"},
    {"\u0492", "CYRILLIC CAPITAL LETTER GHE WITH STROKE"},
    {"\u0493", "CYRILLIC SMALL LETTER GHE WITH STROKE"},
    {"\u0494", "CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK"},
    {"\u0495", "CYRILLIC SMALL LETTER GHE WITH MIDDLE HOOK"},
    {"\u0496", "CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER"},
    {"\u0497", "CYRILLIC SMALL LETTER ZHE WITH DESCENDER"},
    {"\u0498", "CYRILLIC CAPITAL LETTER ZE WITH DESCENDER"},
    {"\u0499", "CYRILLIC SMALL LETTER ZE WITH DESCENDER"},
    {"\u049A", "CYRILLIC CAPITAL LETTER KA WITH DESCENDER"},
    {"\u049B", "CYRILLIC SMALL LETTER KA WITH DESCENDER"},
    {"\u049C", "CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE"},
    {"\u049D", "CYRILLIC SMALL LETTER KA WITH VERTICAL STROKE"},
    {"\u049E", "CYRILLIC CAPITAL LETTER KA WITH STROKE"},
    {"\u049F", "CYRILLIC SMALL LETTER KA WITH STROKE"},
    {"\u04A0", "CYRILLIC CAPITAL LETTER BASHKIR KA"},
    {"\u04A1", "CYRILLIC SMALL LETTER BASHKIR KA"},
    {"\u04A2", "CYRILLIC CAPITAL LETTER EN WITH DESCENDER"},
    {"\u04A3", "CYRILLIC SMALL LETTER EN WITH DESCENDER"},
    {"\u04A4", "CYRILLIC CAPITAL LIGATURE EN GHE"},
    {"\u04A5", "CYRILLIC SMALL LIGATURE EN GHE"},
    {"\u04A6", "CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK"},
    {"\u04A7", "CYRILLIC SMALL LETTER PE WITH MIDDLE HOOK"},
    {"\u04A8", "CYRILLIC CAPITAL LETTER ABKHASIAN HA"},
    {"\u04A9", "CYRILLIC SMALL LETTER ABKHASIAN HA"},
    {"\u04AA", "CYRILLIC CAPITAL LETTER ES WITH DESCENDER"},
    {"\u04AB", "CYRILLIC SMALL LETTER ES WITH DESCENDER"},
    {"\u04AC", "CYRILLIC CAPITAL LETTER TE WITH DESCENDER"},
    {"\u04AD", "CYRILLIC SMALL LETTER TE WITH DESCENDER"},
    {"\u04AE", "CYRILLIC CAPITAL LETTER STRAIGHT U"},
    {"\u04AF", "CYRILLIC SMALL LETTER STRAIGHT U"},
    {"\u04B0", "CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE"},
    {"\u04B1", "CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE"},
    {"\u04B2", "CYRILLIC CAPITAL LETTER HA WITH DESCENDER"},
    {"\u04B3", "CYRILLIC SMALL LETTER HA WITH DESCENDER"},
    {"\u04B4", "CYRILLIC CAPITAL LIGATURE TE TSE"},
    {"\u04B5", "CYRILLIC SMALL LIGATURE TE TSE"},
    {"\u04B6", "CYRILLIC CAPITAL LETTER CHE WITH DESCENDER"},
    {"\u04B7", "CYRILLIC SMALL LETTER CHE WITH DESCENDER"},
    {"\u04B8", "CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE"},
    {"\u04B9", "CYRILLIC SMALL LETTER CHE WITH VERTICAL STROKE"},
    {"\u04BA", "CYRILLIC CAPITAL LETTER SHHA"},
    {"\u04BB", "CYRILLIC SMALL LETTER SHHA"},
    {"\u04BC", "CYRILLIC CAPITAL LETTER ABKHASIAN CHE"},
    {"\u04BD", "CYRILLIC SMALL LETTER ABKHASIAN CHE"},
    {"\u04BE", "CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER"},
    {"\u04BF", "CYRILLIC SMALL LETTER ABKHASIAN CHE WITH DESCENDER"},
    {"\u04C0", "CYRILLIC LETTER PALOCHKA"},
    {"\u04C1", "CYRILLIC CAPITAL LETTER ZHE WITH BREVE"},
    {"\u04C2", "CYRILLIC SMALL LETTER ZHE WITH BREVE"},
    {"\u04C3", "CYRILLIC CAPITAL LETTER KA WITH HOOK"},
    {"\u04C4", "CYRILLIC SMALL LETTER KA WITH HOOK"},
    {"\u04C7", "CYRILLIC CAPITAL LETTER EN WITH HOOK"},
    {"\u04C8", "CYRILLIC SMALL LETTER EN WITH HOOK"},
    {"\u04CB", "CYRILLIC CAPITAL LETTER KHAKASSIAN CHE"},
    {"\u04CC", "CYRILLIC SMALL LETTER KHAKASSIAN CHE"},
    {"\u04D0", "CYRILLIC CAPITAL LETTER A WITH BREVE"},
    {"\u04D1", "CYRILLIC SMALL LETTER A WITH BREVE"},
    {"\u04D2", "CYRILLIC CAPITAL LETTER A WITH DIAERESIS"},
    {"\u04D3", "CYRILLIC SMALL LETTER A WITH DIAERESIS"},
    {"\u04D4", "CYRILLIC CAPITAL LIGATURE A IE"},
    {"\u04D5", "CYRILLIC SMALL LIGATURE A IE"},
    {"\u04D6", "CYRILLIC CAPITAL LETTER IE WITH BREVE"},
    {"\u04D7", "CYRILLIC SMALL LETTER IE WITH BREVE"},
    {"\u04D8", "CYRILLIC CAPITAL LETTER SCHWA"},
    {"\u04D9", "CYRILLIC SMALL LETTER SCHWA"},
    {"\u04DA", "CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS"},
    {"\u04DB", "CYRILLIC SMALL LETTER SCHWA WITH DIAERESIS"},
    {"\u04DC", "CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS"},
    {"\u04DD", "CYRILLIC SMALL LETTER ZHE WITH DIAERESIS"},
    {"\u04DE", "CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS"},
    {"\u04DF", "CYRILLIC SMALL LETTER ZE WITH DIAERESIS"},
    {"\u04E0", "CYRILLIC CAPITAL LETTER ABKHASIAN DZE"},
    {"\u04E1", "CYRILLIC SMALL LETTER ABKHASIAN DZE"},
    {"\u04E2", "CYRILLIC CAPITAL LETTER I WITH MACRON"},
    {"\u04E3", "CYRILLIC SMALL LETTER I WITH MACRON"},
    {"\u04E4", "CYRILLIC CAPITAL LETTER I WITH DIAERESIS"},
    {"\u04E5", "CYRILLIC SMALL LETTER I WITH DIAERESIS"},
    {"\u04E6", "CYRILLIC CAPITAL LETTER O WITH DIAERESIS"},
    {"\u04E7", "CYRILLIC SMALL LETTER O WITH DIAERESIS"},
    {"\u04E8", "CYRILLIC CAPITAL LETTER BARRED O"},
    {"\u04E9", "CYRILLIC SMALL LETTER BARRED O"},
    {"\u04EA", "CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS"},
    {"\u04EB", "CYRILLIC SMALL LETTER BARRED O WITH DIAERESIS"},
    {"\u04EE", "CYRILLIC CAPITAL LETTER U WITH MACRON"},
    {"\u04EF", "CYRILLIC SMALL LETTER U WITH MACRON"},
    {"\u04F0", "CYRILLIC CAPITAL LETTER U WITH DIAERESIS"},
    {"\u04F1", "CYRILLIC SMALL LETTER U WITH DIAERESIS"},
    {"\u04F2", "CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE"},
    {"\u04F3", "CYRILLIC SMALL LETTER U WITH DOUBLE ACUTE"},
    {"\u04F4", "CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS"},
    {"\u04F5", "CYRILLIC SMALL LETTER CHE WITH DIAERESIS"},
    {"\u04F8", "CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS"},
    {"\u04F9", "CYRILLIC SMALL LETTER YERU WITH DIAERESIS"},
  };
  
}
